(*************************************************************

 Addict (c) Addictive Software
 Contact: addictsw@kagi.com or support@addictivesoftware.com

 TRichView (c) Sergey Tkachenko
 Contact: svt@trichview.com

 TCustomRichViewEdit (11.0+) parser for Addict 3 and 4

 History:

 2011-Mar-18 - Skipping hidden text when checking
 2009-Jan-16 - Compatibility with D2009 and Addict 4
 2008-Jun-23 - Rewrite with new TRVSerializer class (TRichView 10.3+).
 Fix for bug when starting checking from before/end of table.
 2007-Sep-09 - RVDONOTUSEUNICODE compiler define is supported.
 2005-Sep-25 - Fix in thesaurus word repaint. SelectChangedWord.
 2005-Sep-07 - Fix (related to ctSmart)
 2004-Aug-27 - Fix
 2004-Aug-15 - Different method of text replacing
 2002-Jan-31 - Spell checking Unicode
 2001-Sep-29 - Update for RichView 1.5.19 (spellcheck includes tables)
 2001-Jun-30 - Update for RichView 1.5.4
 2001-Aug-01 - Initial Write

 **************************************************************)

unit ad3RichView;

{$I addict3.inc}
{$I RV_Defs.inc}

interface

uses
  windows, sysutils, classes, controls, messages,
  ad3ParserBase, RichView, RVEdit, RVSer, CRVData, CRVFData, RVERVData,
  RVScroll,
  RVStyle, RVItem, RVUni, RVTypes, RVTable, RVRVData;

type

  TRichViewParser3 = class(TControlParser)
  protected
    FCodePage:                                        TRVCodePage;
    FEdit:                                            TCustomRichViewEdit;
    FCurAbsItemNo, FCurOffs, FEndAbsItemNo, FEndOffs: Integer;
    FUsingEnd:                                        Boolean;

    FLine:       String;
    FLineIndex:  Integer;
    FLineLength: Integer;
    FDirty:      Boolean;
    FSerializer: TRVSerializer;
    FCleaned:    Boolean;

    FSelectChangedWord: Boolean;
    procedure EnsureSerializer;
    function EnsureLineCurrent: Boolean; virtual;
    function CanCheckItem(RVData: TCustomRVData; ItemNo: Integer): Boolean;

  public
    destructor Destroy; override;
    procedure Initialize(EditControl: Pointer); override;
    function GetChar: char; override;
    function GetLine: String; override;
    function MoveNext: Boolean; override;
    function MovePrevious: Boolean; override;
    procedure SetPosition(XPos: LongInt; YPos: LongInt;
      PosType: TPositionType); override;
    procedure GetPosition(var XPos: LongInt; var YPos: LongInt); override;
    procedure SelectWord(Length: LongInt); override;
    procedure ReplaceWord(Replacement: String; State: LongInt); override;
    procedure IgnoreWord(State: LongInt); override;
    procedure CenterSelection; override;
    procedure GetCursorPosition(var XPos: LongInt; var YPos: LongInt); override;
    procedure GetSelectionPosition(var XPosStart: LongInt;
      var YPosStart: LongInt; var XPosEnd: LongInt;
      var YPosEnd: LongInt); override;
    procedure GetControlScreenPosition(var ControlPosition: TRect); override;
    procedure GetSelectionScreenPosition(var SelectionPosition: TRect);
      override;
    procedure UndoLast(UndoState: LongInt; UndoAction: LongInt;
      var UndoData: LongInt); override;
    procedure Cleanup;
    property CodePage: TRVCodePage write FCodePage;
    property SelectChangedWord: Boolean write FSelectChangedWord;
    property CurAbsItemNo: Integer read FCurAbsItemNo;
  end;

implementation

{------------------------------- Utility Functions ----------------------------}
function TRichViewParser3.CanCheckItem(RVData: TCustomRVData;
  ItemNo: Integer): Boolean;
var
  StyleNo: Integer;
begin
  StyleNo := RVData.GetRVData.GetItemStyle(ItemNo);
  Result := (StyleNo >= 0) and (FEdit.Style.TextStyles[StyleNo].Charset <>
    SYMBOL_CHARSET) and not(rvprModifyProtect in FEdit.Style.TextStyles[StyleNo]
    .Protection);
  if Result and not(rvoShowHiddenText in FEdit.Options) then
  begin
    Result := not(rvteoHidden in FEdit.Style.TextStyles[StyleNo].Options);
    while RVData <> FEdit.RVData do
    begin
      if RVData.GetSourceRVData is TRVTableCellData then
      begin
        Result := not TRVTableCellData(RVData.GetSourceRVData).GetTable.Hidden;
        if not Result then
          exit;
      end;
      RVData := RVData.GetAbsoluteParentData;
    end;
  end;
end;

{------------------------------------------------------------------------------}
// Converting text from internal representation to String
function ConvertItemTextToString(const ItemText: TRVRawByteString;
  UnicodeItem: Boolean; CodePage: Cardinal): String;
begin
  {$IFDEF RVUNICODESTR} // <-- declared in RV_Defs.inc
  // Delphi 2009+: String is Unicode
  if UnicodeItem then
    Result := RVU_RawUnicodeToWideString(ItemText)
  else
    Result := RVU_RawUnicodeToWideString(RVU_AnsiToUnicode(CodePage, ItemText));
  {$ELSE}
  // Delphi 4-2007: String is ANSI
  {$IFNDEF RVDONOTUSEUNICODE}
  if UnicodeItem then
    Result := RVU_UnicodeToAnsi(CodePage, ItemText)
  else
    {$ENDIF}
    Result := ItemText;
  {$ENDIF}
end;

// Converting text from String to internal representation
function ConvertStringToItemText(const Text: String; UnicodeItem: Boolean;
  CodePage: Cardinal): TRVRawByteString;
begin
  {$IFDEF RVUNICODESTR} // <-- declared in RV_Defs.inc
  if UnicodeItem then
    Result := RVU_GetRawUnicode(Text)
  else
    Result := TRVAnsiString(Text);
  {$ELSE}
  {$IFNDEF RVDONOTUSEUNICODE}
  if UnicodeItem then
    Result := RVU_AnsiToUnicode(CodePage, Text)
  else
    {$ENDIF}
    Result := Text;
  {$ENDIF}
end;

{------------------------------------------------------------------------------}
function TRichViewParser3.EnsureLineCurrent: Boolean;
{...............................................}
  procedure SkipForward;
  var
    ItemNo:    Integer;
    RVData:    TCustomRVData;
    AfterItem: Boolean;
  begin
    while FCurAbsItemNo < FSerializer.AbsItemCount do
    begin
      FSerializer.AbsoluteToRV(FCurAbsItemNo, RVData, ItemNo, AfterItem);
      if CanCheckItem(RVData, ItemNo) then
        exit;
      inc(FCurAbsItemNo);
    end;
  end;
{...............................................}
  procedure SkipBackward;
  var
    ItemNo:    Integer;
    RVData:    TCustomRVData;
    AfterItem: Boolean;
  begin
    while FCurAbsItemNo >= 0 do
    begin
      FSerializer.AbsoluteToRV(FCurAbsItemNo, RVData, ItemNo, AfterItem);
      if CanCheckItem(RVData, ItemNo) then
        exit;
      dec(FCurAbsItemNo);
    end;
  end;

{...............................................}
var
  r, AfterItem: Boolean;
  ItemNo:       Integer;
  RVData:       TCustomRVData;
  LCodePage:    TRVCodePage;
begin
  EnsureSerializer;
  Result := True;
  r := True;

  // Auto-Skip to another line if needbe

  if (FCurAbsItemNo = FLineIndex) then
  begin
    if (FCurOffs >= FLineLength) then
    begin
      inc(FCurAbsItemNo);
      SkipForward;
      FCurOffs := 0;
    end
    else if (FCurOffs < 0) then
    begin
      dec(FCurAbsItemNo);
      SkipBackward;
      FCurOffs := -1;
    end;
  end;
  if FCurAbsItemNo < 0 then
  begin
    FCurAbsItemNo := 0;
    FCurOffs := 0;
    r := False;
  end;
  SkipForward;
  // Now fetch the new line if needbe

  if FCurAbsItemNo >= FSerializer.AbsItemCount then
    r := False;
  if (FCurAbsItemNo <> FLineIndex) or FDirty then
  begin
    Result := (FCurAbsItemNo < FSerializer.AbsItemCount) and
      (FCurAbsItemNo >= 0);
    if Result then
    begin
      FSerializer.AbsoluteToRV(FCurAbsItemNo, RVData, ItemNo, AfterItem);
      FDirty := False;
      {$IFDEF RVUNICODESTR}
      LCodePage := RVData.GetRVData.GetItemCodePage(ItemNo);
      {$ELSE}
      LCodePage := FCodePage;
      {$ENDIF}
      FLine := ConvertItemTextToString(RVData.GetRVData.GetItemTextR(ItemNo),
        {$IFNDEF RVDONOTUSEUNICODE}
        rvioUnicode in RVData.GetRVData.GetItem(ItemNo).ItemOptions,
        {$ELSE}
        False,
        {$ENDIF}
        LCodePage) + #13#10;
      FLineLength := Length(FLine);
      FLineIndex := FCurAbsItemNo;
      if (FCurOffs = -1) then
        FCurOffs := FLineLength - 1;
    end;
  end;
  Result := Result and r;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.EnsureSerializer;
begin
  if FSerializer = nil then
    FSerializer := TRVSerializer.Create(FEdit.RVData);
end;

{--------------------------      Parser Implementation    ---------------------}
destructor TRichViewParser3.Destroy;
begin
  FSerializer.Free;
  inherited;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.Initialize(EditControl: Pointer);
begin
  FSerializer.Free;
  FEdit := TObject(EditControl) as TCustomRichViewEdit;
  FSerializer := TRVSerializer.Create(FEdit.RVData);
  FUsingEnd := False;
  FCurAbsItemNo := -1;
  FCurOffs := 0;
  FDirty := True;
  FLineLength := 0;
  FLineIndex := -1;
  FLine := '';
end;

{------------------------------------------------------------------------------}
function TRichViewParser3.GetChar: char;
begin
  if (EnsureLineCurrent) then
    Result := FLine[FCurOffs + 1]
  else
    Result := #0;
end;

{------------------------------------------------------------------------------}
function TRichViewParser3.GetLine: String;
begin
  if (EnsureLineCurrent) then
    Result := FLine
  else
    Result := '';
end;

{------------------------------------------------------------------------------}
function TRichViewParser3.MoveNext: Boolean;
begin
  inc(FCurOffs);
  Result := EnsureLineCurrent;
  if FUsingEnd and Result then
  begin
    if ((FCurAbsItemNo > FEndAbsItemNo) or ((FCurOffs > FEndOffs) and
      (FCurAbsItemNo = FEndAbsItemNo))) then
      Result := False;
  end;
end;

{------------------------------------------------------------------------------}
function TRichViewParser3.MovePrevious: Boolean;
begin
  dec(FCurOffs);
  Result := EnsureLineCurrent;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.SetPosition(XPos: LongInt; YPos: LongInt;
  PosType: TPositionType);
begin
  if (PosType = ptCurrent) then
  begin
    FDirty := True;
    FCurOffs := XPos;
    FCurAbsItemNo := YPos;
  end
  else if (PosType = ptEnd) then
  begin
    FUsingEnd := True;
    FEndOffs := XPos;
    FEndAbsItemNo := YPos;
  end;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.GetPosition(var XPos: LongInt; var YPos: LongInt);
begin
  if FCleaned then
  begin
    FCleaned := False;
    GetCursorPosition(XPos, YPos);
  end
  else
  begin
    YPos := FCurAbsItemNo;
    XPos := FCurOffs;
  end;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.SelectWord(Length: LongInt);
var
  ItemNo:    Integer;
  RVData:    TCustomRVData;
  AfterItem: Boolean;
begin
  EnsureSerializer;
  if (FCurAbsItemNo >= FSerializer.AbsItemCount) or (FCurAbsItemNo < 0) then
    exit;
  FSerializer.AbsoluteToRV(FCurAbsItemNo, RVData, ItemNo, AfterItem);
  RVData := TCustomRVFormattedData(RVData.GetRVData);
  dec(FCurOffs, Length);
  if RVData.GetItemStyle(ItemNo) < 0 then
  begin
    if FCurOffs < RVData.GetOffsBeforeItem(ItemNo) then
      FCurOffs := RVData.GetOffsBeforeItem(ItemNo);
    if FCurOffs > RVData.GetOffsAfterItem(ItemNo) then
      FCurOffs := RVData.GetOffsAfterItem(ItemNo);
    if FCurOffs + Length > RVData.GetOffsAfterItem(ItemNo) then
      Length := RVData.GetOffsAfterItem(ItemNo) - FCurOffs;
    RVData := RVData.Edit;
    TCustomRVFormattedData(RVData).SetSelectionBounds(ItemNo, FCurOffs, ItemNo,
      FCurOffs + Length);
  end
  else
  begin
    if FCurOffs + 1 < RVData.GetOffsBeforeItem(ItemNo) then
      FCurOffs := RVData.GetOffsBeforeItem(ItemNo) - 1;
    if FCurOffs + 1 > RVData.GetOffsAfterItem(ItemNo) then
      FCurOffs := RVData.GetOffsAfterItem(ItemNo) - 1;
    if FCurOffs + Length + 1 > RVData.GetOffsAfterItem(ItemNo) then
      Length := RVData.GetOffsAfterItem(ItemNo) - FCurOffs - 1;
    RVData := RVData.Edit;
    TCustomRVFormattedData(RVData).SetSelectionBounds(ItemNo, FCurOffs + 1,
      ItemNo, FCurOffs + Length + 1);
  end;
  TCustomRVFormattedData(RVData).Invalidate;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.ReplaceWord(Replacement: String; State: LongInt);
var
  s, rep:                       TRVRawByteString;
  ItemNo, StyleNo, len, replen: Integer;
begin
  len := Length(FEdit.GetSelText);
  if FUsingEnd and (FCurAbsItemNo = FEndAbsItemNo) then
    inc(FEndOffs, Length(Replacement) - len);
  // This code is used instead of InsertText to keep item properties
  ItemNo := FEdit.TopLevelEditor.CurItemNo;
  StyleNo := FEdit.TopLevelEditor.GetItemStyle(ItemNo);
  s := FEdit.GetCurrentItemTextR;
  rep := ConvertStringToItemText(Replacement,
    {$IFNDEF RVDONOTUSEUNICODE}
    FEdit.Style.TextStyles[StyleNo].Unicode,
    {$ELSE}
    False,
    {$ENDIF}
    FEdit.TopLevelEditor.RVData.GetStyleCodePage(StyleNo));
  {$IFNDEF RVDONOTUSEUNICODE}
  if FEdit.Style.TextStyles[StyleNo].Unicode then
  begin
    Delete(s, FCurOffs * 2 + 1, len * 2);
    Insert(rep, s, FCurOffs * 2 + 1);
    replen := Length(rep) div 2;
  end
  else
  {$ENDIF}
  begin
    Delete(s, FCurOffs + 1, len);
    Insert(rep, s, FCurOffs + 1);
    replen := Length(rep);
  end;
  FEdit.SetCurrentItemTextR(s);
  if FSelectChangedWord then
    FEdit.TopLevelEditor.SetSelectionBounds(ItemNo, FCurOffs + 1, ItemNo,
      FCurOffs + replen + 1)
  else
    FEdit.TopLevelEditor.SetSelectionBounds(ItemNo, FCurOffs + replen + 1,
      ItemNo, FCurOffs + replen + 1);
  FEdit.TopLevelEditor.Invalidate;
  // FEdit.InsertText(Replacement);
  inc(FCurOffs, replen);
  FDirty := True;
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.IgnoreWord(State: LongInt);
begin
  inc(FCurOffs, Length(FEdit.GetSelText));
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.CenterSelection;
begin

end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.GetCursorPosition(var XPos: LongInt;
  var YPos: LongInt);
var
  ItemNo:    Integer;
  RVData:    TCustomRVData;
  AfterItem: Boolean;
begin
  EnsureSerializer;
  RVData := FEdit.TopLevelEditor.RVData;
  ItemNo := FEdit.TopLevelEditor.CurItemNo;
  XPos := FEdit.TopLevelEditor.OffsetInCurItem;
  AfterItem := XPos >= RVData.GetOffsAfterItem(ItemNo);
  if RVData.GetItemStyle(ItemNo) >= 0 then
    dec(XPos);
  FSerializer.RVToAbsolute(RVData, ItemNo, AfterItem, YPos);
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.GetSelectionPosition(var XPosStart: LongInt;
  var YPosStart: LongInt; var XPosEnd: LongInt; var YPosEnd: LongInt);
var
  ItemNo1, ItemNo2:       Integer;
  RVData:                 TCustomRVData;
  AfterItem1, AfterItem2: Boolean;

begin
  EnsureSerializer;
  RVData := FEdit.TopLevelEditor.RVData;
  FEdit.TopLevelEditor.RVData.GetSelectionBoundsEx(ItemNo1, XPosStart, ItemNo2,
    XPosEnd, True);
  AfterItem1 := XPosStart >= RVData.GetOffsAfterItem(ItemNo1);
  AfterItem2 := XPosEnd >= RVData.GetOffsAfterItem(ItemNo2);
  if RVData.GetItemStyle(ItemNo1) >= 0 then
    dec(XPosStart);
  if RVData.GetItemStyle(ItemNo2) >= 0 then
    dec(XPosEnd);
  FSerializer.RVToAbsolute(RVData, ItemNo1, AfterItem1, YPosStart);
  if (ItemNo1 = ItemNo2) and (XPosStart = XPosEnd) then
    YPosEnd := YPosStart
  else
    FSerializer.RVToAbsolute(RVData, ItemNo2, AfterItem2, YPosEnd);
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.GetControlScreenPosition(var ControlPosition: TRect);
var
  P: TPoint;
begin
  P := Point(0, 0);
  P := FEdit.ClientToScreen(P);
  ControlPosition := Rect(P.X, P.Y, P.X + FEdit.Width, P.Y + FEdit.Height);
end;

{------------------------------------------------------------------------------}
procedure TRichViewParser3.GetSelectionScreenPosition
  (var SelectionPosition: TRect);
begin
  SelectionPosition := FEdit.GetSelectionRect;
end;

// ************************************************************

procedure TRichViewParser3.UndoLast(UndoState: LongInt; UndoAction: LongInt;
  var UndoData: LongInt);
begin

end;

// ************************************************************
procedure TRichViewParser3.Cleanup;
begin
  FSerializer.Free;
  FSerializer := nil;
  FCurAbsItemNo := -1;
  FCurOffs := 0;
  FLineIndex := -1;
  FLine := '';
  FCleaned := True;
end;

end.
